<?php
/**
 * Plugin Name: CodeGrinders Auto Poster AI
 * Plugin URI: https://plugins.codegrinders.tech/codegrinders-auto-poster-ai
 * Description: AI-powered auto-posting plugin that generates content and images based on keywords and knowledge base.
 * Version: 1.0.0
 * Author: CodeGrinders
 * Author URI: https://codegrinders.tech
 * Text Domain: codegrinders-auto-poster-ai
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('CGAP_PLUGIN_URL', plugin_dir_url(__FILE__));
define('CGAP_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('CGAP_VERSION', '1.0.0');
define('CGAP_PLUGIN_SLUG', 'codegrinders-auto-poster-ai');

/**
 * Main Auto Poster Plugin Class
 */
class CodeGrindersAutoPoster {
    
    public function __construct() {
        add_action('init', array($this, 'init'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));
        add_action('wp_ajax_cgap_save_settings', array($this, 'save_settings'));
        add_action('wp_ajax_cgap_validate_license', array($this, 'validate_license'));
        add_action('wp_ajax_cgap_check_license', array($this, 'ajax_check_license'));
        add_action('wp_ajax_cgap_generate_post', array($this, 'generate_post'));
        add_action('wp_ajax_cgap_generate_image', array($this, 'generate_image'));
        add_action('wp_ajax_cgap_upload_knowledge', array($this, 'upload_knowledge'));
        add_action('admin_footer', array($this, 'powered_by_footer'));
        add_action('cgap_scheduled_post', array($this, 'create_scheduled_post'));
        
        // Schedule cron job
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    /**
     * Initialize plugin
     */
    public function init() {
        // Create database tables if needed
        $this->create_tables();
        
        // Ensure plugin exists in license database
        $this->ensure_plugin_in_database();
        
        // Set default options
        if (!get_option('cgap_settings')) {
            $default_settings = array(
                'license_key' => '',
                'license_status' => 'inactive',
                'ai_api_key' => '',
                'ai_model' => 'gpt-3.5-turbo',
                'keywords' => '',
                'knowledge_base' => '',
                'character_count' => 0,
                'max_characters_free' => 5000,
                'post_frequency' => 'weekly',
                'post_status' => 'draft',
                'post_category' => '',
                'post_tags' => '',
                'generate_images' => false,
                'image_style' => 'realistic',
                'last_post_date' => '',
                'posts_this_week' => 0,
                'is_enabled' => false
            );
            add_option('cgap_settings', $default_settings);
        }
    }
    
    /**
     * Ensure plugin exists in license database
     */
    private function ensure_plugin_in_database() {
        $config_path = CGAP_PLUGIN_PATH . '../../config/config.php';
        
        if (!file_exists($config_path)) {
            error_log('CGAP: Config file not found, skipping database check');
            return;
        }
        
        try {
            $config = include $config_path;
            $pdo = new PDO(
                "mysql:host={$config['db_host']};dbname={$config['db_name']};charset={$config['db_charset']};port={$config['db_port']}",
                $config['db_user'],
                $config['db_pass'],
                array(
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false
                )
            );
            
            // Check if plugin exists
            $plugin_stmt = $pdo->prepare("SELECT id FROM plugins WHERE folder = 'codegrinders-auto-poster-ai'");
            $plugin_stmt->execute();
            $plugin = $plugin_stmt->fetch();
            
            if (!$plugin) {
                // Add plugin to database
                $insert_stmt = $pdo->prepare("INSERT INTO plugins (name, description, folder) VALUES (?, ?, ?)");
                $insert_stmt->execute([
                    'CodeGrinders Auto Poster AI',
                    'AI-powered automatic content generation and posting plugin for WordPress. Generate high-quality blog posts automatically using OpenAI\'s GPT models. Features include knowledge base integration, AI image generation, and automatic scheduling.',
                    'codegrinders-auto-poster-ai'
                ]);
                error_log('CGAP: Plugin added to license database');
            }
            
        } catch (Exception $e) {
            error_log('CGAP: Error ensuring plugin in database: ' . $e->getMessage());
        }
    }
    
    /**
     * Create database tables
     */
    private function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Auto posts table
        $table_name = $wpdb->prefix . 'cgap_auto_posts';
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            post_id bigint(20) NOT NULL,
            keywords text NOT NULL,
            generated_content text NOT NULL,
            image_url varchar(500) NULL,
            generation_date datetime DEFAULT CURRENT_TIMESTAMP,
            status varchar(50) DEFAULT 'published',
            PRIMARY KEY (id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Activate plugin
     */
    public function activate() {
        // Schedule cron jobs
        if (!wp_next_scheduled('cgap_daily_check')) {
            wp_schedule_event(time(), 'daily', 'cgap_daily_check');
        }
        
        if (!wp_next_scheduled('cgap_license_check')) {
            wp_schedule_event(time(), 'weekly', 'cgap_license_check');
        }
        
        // Add cron hooks
        add_action('cgap_daily_check', array($this, 'check_scheduled_posts'));
        add_action('cgap_license_check', array($this, 'check_license_status'));
    }
    
    /**
     * Deactivate plugin
     */
    public function deactivate() {
        // Clear scheduled cron jobs
        wp_clear_scheduled_hook('cgap_daily_check');
        wp_clear_scheduled_hook('cgap_license_check');
    }
    
    /**
     * Check for scheduled posts
     */
    public function check_scheduled_posts() {
        $settings = get_option('cgap_settings', array());
        
        if (!$settings['is_enabled']) {
            return;
        }
        
        $is_license_active = $this->is_license_active();
        $frequency = $settings['post_frequency'] ?? 'weekly';
        $last_post_date = $settings['last_post_date'] ?? '';
        $posts_this_week = $settings['posts_this_week'] ?? 0;
        
        $should_post = false;
        
        // Check if it's time to post
        switch ($frequency) {
            case 'daily':
                if ($is_license_active) {
                    $should_post = empty($last_post_date) || 
                                  (strtotime($last_post_date) < strtotime('-1 day'));
                }
                break;
                
            case 'weekly':
                if ($is_license_active) {
                    $should_post = empty($last_post_date) || 
                                  (strtotime($last_post_date) < strtotime('-1 week'));
                } else {
                    // Free version: only 1 post per week
                    $should_post = $posts_this_week < 1 && 
                                  (empty($last_post_date) || 
                                   (strtotime($last_post_date) < strtotime('-1 week')));
                }
                break;
                
            case 'monthly':
                if ($is_license_active) {
                    $should_post = empty($last_post_date) || 
                                  (strtotime($last_post_date) < strtotime('-1 month'));
                }
                break;
        }
        
        if ($should_post) {
            $this->create_scheduled_post();
        }
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            'Auto Poster AI',
            'Auto Poster AI',
            'manage_options',
            'auto-poster-ai',
            array($this, 'dashboard_page'),
            'dashicons-admin-post',
            31
        );
        
        add_submenu_page(
            'auto-poster-ai',
            'Dashboard',
            'Dashboard',
            'manage_options',
            'auto-poster-ai',
            array($this, 'dashboard_page')
        );
        
        add_submenu_page(
            'auto-poster-ai',
            'Settings',
            'Settings',
            'manage_options',
            'auto-poster-ai-settings',
            array($this, 'settings_page')
        );
        
        add_submenu_page(
            'auto-poster-ai',
            'Knowledge Base',
            'Knowledge Base',
            'manage_options',
            'auto-poster-ai-knowledge',
            array($this, 'knowledge_page')
        );
        
        add_submenu_page(
            'auto-poster-ai',
            'Generated Posts',
            'Generated Posts',
            'manage_options',
            'auto-poster-ai-posts',
            array($this, 'posts_page')
        );
    }
    
    /**
     * Enqueue admin scripts
     */
    public function admin_scripts($hook) {
        if (strpos($hook, 'auto-poster-ai') !== false) {
            wp_enqueue_style('cgap-admin', CGAP_PLUGIN_URL . 'assets/admin.css', [], CGAP_VERSION);
            wp_enqueue_script('cgap-admin', CGAP_PLUGIN_URL . 'assets/admin.js', array('jquery'), CGAP_VERSION, true);
            wp_localize_script('cgap-admin', 'cgap_ajax', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('cgap_nonce')
            ));
        }
    }
    
    /**
     * Dashboard page
     */
    public function dashboard_page() {
        $settings = get_option('cgap_settings', array());
        $is_license_active = $this->is_license_active();
        $character_count = $settings['character_count'] ?? 0;
        $max_characters = $settings['max_characters_free'] ?? 5000;
        $usage_percentage = ($character_count / $max_characters) * 100;
        
        include CGAP_PLUGIN_PATH . 'admin/dashboard.php';
    }
    
    /**
     * Settings page
     */
    public function settings_page() {
        $settings = get_option('cgap_settings', array());
        include CGAP_PLUGIN_PATH . 'admin/settings.php';
    }
    
    /**
     * Knowledge base page
     */
    public function knowledge_page() {
        $settings = get_option('cgap_settings', array());
        include CGAP_PLUGIN_PATH . 'admin/knowledge.php';
    }
    
    /**
     * Posts page
     */
    public function posts_page() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'cgap_auto_posts';
        
        $page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $per_page = 20;
        $offset = ($page - 1) * $per_page;
        
        $total_items = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
        $total_pages = ceil($total_items / $per_page);
        
        $posts = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name ORDER BY generation_date DESC LIMIT %d OFFSET %d",
            $per_page, $offset
        ));
        
        include CGAP_PLUGIN_PATH . 'admin/posts.php';
    }
    
    /**
     * Check if license is active
     */
    public function is_license_active() {
        $settings = get_option('cgap_settings', array());
        $license_status = $settings['license_status'] ?? 'inactive';
        
        if ($license_status !== 'active') {
            return false;
        }
        
        // Check if license has expired
        $expires_at = $settings['license_expires_at'] ?? null;
        if ($expires_at && strtotime($expires_at) < time()) {
            // Update status to expired
            $settings['license_status'] = 'expired';
            update_option('cgap_settings', $settings);
            return false;
        }
        
        return true;
    }
    
    /**
     * Check license status with database
     */
    public function check_license_status() {
        $settings = get_option('cgap_settings', array());
        $license_key = $settings['license_key'] ?? '';
        
        if (empty($license_key)) {
            return false;
        }
        
        $result = $this->validate_license_with_server($license_key);
        
        if ($result['valid']) {
            $settings['license_status'] = 'active';
            update_option('cgap_settings', $settings);
            return true;
        } else {
            $settings['license_status'] = 'inactive';
            update_option('cgap_settings', $settings);
            return false;
        }
    }
    
    /**
     * Check license via AJAX
     */
    public function ajax_check_license() {
        check_ajax_referer('cgap_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $result = $this->check_license_status();
        
        if ($result) {
            wp_send_json_success('License is valid and active.');
        } else {
            wp_send_json_error('License check failed. Please verify your license key.');
        }
    }
    
    /**
     * Get license expiration info
     */
    public function get_license_expiration_info() {
        $settings = get_option('cgap_settings', array());
        $expires_at = $settings['license_expires_at'] ?? null;
        
        if (!$expires_at) {
            return array('has_expiration' => false);
        }
        
        $expiration_time = strtotime($expires_at);
        $current_time = time();
        $days_until_expiry = ceil(($expiration_time - $current_time) / (24 * 60 * 60));
        
        return array(
            'has_expiration' => true,
            'expires_at' => $expires_at,
            'days_until_expiry' => $days_until_expiry,
            'is_expired' => $days_until_expiry <= 0,
            'is_expiring_soon' => $days_until_expiry <= 30 && $days_until_expiry > 0
        );
    }
    
    /**
     * Get character limit based on license
     */
    public function get_character_limit() {
        if ($this->is_license_active()) {
            return PHP_INT_MAX; // Unlimited for paid
        }
        return 5000; // Free limit
    }
    
    /**
     * Save settings via AJAX
     */
    public function save_settings() {
        check_ajax_referer('cgap_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $settings = get_option('cgap_settings', array());
        
        // Update settings
        $settings['keywords'] = sanitize_textarea_field($_POST['keywords'] ?? '');
        $settings['post_frequency'] = sanitize_text_field($_POST['post_frequency'] ?? 'weekly');
        $settings['post_status'] = sanitize_text_field($_POST['post_status'] ?? 'draft');
        $settings['post_category'] = sanitize_text_field($_POST['post_category'] ?? '');
        $settings['post_tags'] = sanitize_text_field($_POST['post_tags'] ?? '');
        $settings['generate_images'] = (bool)($_POST['generate_images'] ?? false);
        $settings['image_style'] = sanitize_text_field($_POST['image_style'] ?? 'realistic');
        $settings['ai_api_key'] = sanitize_text_field($_POST['ai_api_key'] ?? '');
        $settings['ai_model'] = sanitize_text_field($_POST['ai_model'] ?? 'gpt-3.5-turbo');
        $settings['is_enabled'] = (bool)($_POST['is_enabled'] ?? false);
        
        update_option('cgap_settings', $settings);
        
        wp_send_json_success('Settings saved successfully!');
    }
    
    /**
     * Validate license via AJAX
     */
    public function validate_license() {
        check_ajax_referer('cgap_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $license_key = sanitize_text_field($_POST['license_key'] ?? '');
        
        if (empty($license_key)) {
            wp_send_json_error('Please enter a license key.');
        }
        
        $result = $this->validate_license_with_server($license_key);
        
        if ($result['valid']) {
            $settings = get_option('cgap_settings', array());
            $settings['license_key'] = $license_key;
            $settings['license_status'] = 'active';
            update_option('cgap_settings', $settings);
            
            wp_send_json_success('License validated successfully! Premium features are now available.');
        } else {
            $settings = get_option('cgap_settings', array());
            $settings['license_status'] = 'inactive';
            update_option('cgap_settings', $settings);
            
            wp_send_json_error('License validation failed: ' . $result['error']);
        }
    }
    
    /**
     * Validate license with server
     */
    public function validate_license_with_server($license_key) {
        // Connect to CodeGrinders database
        $config_path = CGAP_PLUGIN_PATH . '../../config/config.php';
        
        if (!file_exists($config_path)) {
            error_log('CGAP: Config file not found at: ' . $config_path);
            return array('valid' => false, 'error' => 'Configuration file not found. Please contact support.');
        }
        
        $config = include $config_path;
        
        try {
            $pdo = new PDO(
                "mysql:host={$config['db_host']};dbname={$config['db_name']};charset={$config['db_charset']};port={$config['db_port']}",
                $config['db_user'],
                $config['db_pass'],
                array(
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false
                )
            );
            
            // First, get the plugin ID for auto-poster
            $plugin_stmt = $pdo->prepare("SELECT id FROM plugins WHERE folder = 'codegrinders-auto-poster'");
            $plugin_stmt->execute();
            $plugin = $plugin_stmt->fetch();
            
            if (!$plugin) {
                error_log('CGAP: Plugin not found in database');
                return array('valid' => false, 'error' => 'Plugin not found in database. Please contact support.');
            }
            
            // Check if license exists and is valid
            $license_stmt = $pdo->prepare("
                SELECT l.*, u.email 
                FROM licenses l 
                JOIN users u ON l.user_id = u.id 
                WHERE l.license_key = ? AND l.plugin_id = ? AND l.status = 'active'
            ");
            $license_stmt->execute(array($license_key, $plugin['id']));
            $license = $license_stmt->fetch();
            
            if (!$license) {
                error_log('CGAP: Invalid license key: ' . $license_key);
                return array('valid' => false, 'error' => 'Invalid or inactive license key');
            }
            
            // Check if license is expired
            if ($license['expires_at'] && strtotime($license['expires_at']) < time()) {
                error_log('CGAP: License expired for key: ' . $license_key);
                return array('valid' => false, 'error' => 'License has expired');
            }
            
            // Store additional license info
            $settings = get_option('cgap_settings', array());
            $settings['license_user_email'] = $license['email'];
            $settings['license_issued_at'] = $license['issued_at'];
            $settings['license_expires_at'] = $license['expires_at'];
            update_option('cgap_settings', $settings);
            
            error_log('CGAP: License validated successfully for key: ' . $license_key);
            return array('valid' => true, 'error' => '');
            
        } catch (PDOException $e) {
            error_log('CGAP: Database connection failed: ' . $e->getMessage());
            return array('valid' => false, 'error' => 'Database connection failed. Please try again later.');
        } catch (Exception $e) {
            error_log('CGAP: License validation error: ' . $e->getMessage());
            return array('valid' => false, 'error' => 'License validation failed. Please try again.');
        }
    }
    
    /**
     * Upload knowledge base via AJAX with improved validation
     */
    public function upload_knowledge() {
        check_ajax_referer('cgap_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $knowledge_text = sanitize_textarea_field($_POST['knowledge_text'] ?? '');
        $current_text = get_option('cgap_knowledge_base', '');
        
        // Validate input
        if (empty($knowledge_text)) {
            wp_send_json_error('Please enter knowledge base content.');
        }
        
        // Check character limit
        $total_length = strlen($current_text) + strlen($knowledge_text);
        $limit = $this->get_character_limit();
        
        if ($total_length > $limit) {
            $excess = $total_length - $limit;
            wp_send_json_error("Knowledge base exceeds character limit by {$excess} characters. Free version allows up to " . number_format($limit) . " characters. Upgrade to premium for unlimited knowledge base.");
        }
        
        // Clean and optimize knowledge base
        $cleaned_text = $this->clean_knowledge_base($knowledge_text);
        
        // Update knowledge base
        $new_text = $current_text . "\n\n" . $cleaned_text;
        update_option('cgap_knowledge_base', $new_text);
        
        // Update character count
        $settings = get_option('cgap_settings', array());
        $settings['character_count'] = strlen($new_text);
        update_option('cgap_settings', $settings);
        
        wp_send_json_success('Knowledge base updated successfully! Character count: ' . number_format(strlen($new_text)));
    }
    
    /**
     * Clean and optimize knowledge base content
     */
    private function clean_knowledge_base($text) {
        // Remove excessive whitespace
        $text = preg_replace('/\s+/', ' ', $text);
        
        // Remove HTML tags
        $text = strip_tags($text);
        
        // Limit line length for better AI processing
        $lines = explode("\n", $text);
        $cleaned_lines = array();
        
        foreach ($lines as $line) {
            $line = trim($line);
            if (!empty($line)) {
                // Break long lines
                if (strlen($line) > 200) {
                    $words = explode(' ', $line);
                    $current_line = '';
                    foreach ($words as $word) {
                        if (strlen($current_line . ' ' . $word) > 200) {
                            if (!empty($current_line)) {
                                $cleaned_lines[] = $current_line;
                                $current_line = $word;
                            }
                        } else {
                            $current_line .= (empty($current_line) ? '' : ' ') . $word;
                        }
                    }
                    if (!empty($current_line)) {
                        $cleaned_lines[] = $current_line;
                    }
                } else {
                    $cleaned_lines[] = $line;
                }
            }
        }
        
        return implode("\n", $cleaned_lines);
    }
    
    /**
     * Generate post via AJAX
     */
    public function generate_post() {
        check_ajax_referer('cgap_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $settings = get_option('cgap_settings', array());
        $keywords = sanitize_text_field($_POST['keywords'] ?? '');
        
        if (empty($keywords)) {
            wp_send_json_error('Please provide keywords for post generation.');
        }
        
        // Check if free user has exceeded weekly limit
        if (!$this->is_license_active()) {
            $posts_this_week = $settings['posts_this_week'] ?? 0;
            if ($posts_this_week >= 1) {
                wp_send_json_error('Free version is limited to 1 post per week. Upgrade to premium for unlimited posting.');
            }
        }
        
        $result = $this->create_post($keywords);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['error']);
        }
    }
    
    /**
     * Generate image via AJAX
     */
    public function generate_image() {
        check_ajax_referer('cgap_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        if (!$this->is_license_active()) {
            wp_send_json_error('Image generation is a premium feature. Please upgrade to premium.');
        }
        
        $prompt = sanitize_text_field($_POST['prompt'] ?? '');
        $style = sanitize_text_field($_POST['style'] ?? 'realistic');
        
        if (empty($prompt)) {
            wp_send_json_error('Please provide a prompt for image generation.');
        }
        
        $result = $this->generate_ai_image($prompt, $style);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['error']);
        }
    }
    
    /**
     * Create scheduled post
     */
    public function create_scheduled_post() {
        $settings = get_option('cgap_settings', array());
        $keywords = $settings['keywords'] ?? '';
        
        if (empty($keywords)) {
            return;
        }
        
        $result = $this->create_post($keywords);
        
        if ($result['success']) {
            // Update last post date and count
            $settings['last_post_date'] = current_time('mysql');
            if (!$this->is_license_active()) {
                $settings['posts_this_week'] = ($settings['posts_this_week'] ?? 0) + 1;
            }
            update_option('cgap_settings', $settings);
        }
    }
    
    /**
     * Create post with AI content
     */
    private function create_post($keywords) {
        $settings = get_option('cgap_settings', array());
        $api_key = $settings['ai_api_key'] ?? '';
        $model = $settings['ai_model'] ?? 'gpt-3.5-turbo';
        $knowledge_base = get_option('cgap_knowledge_base', '');
        
        if (empty($api_key)) {
            return array('success' => false, 'error' => 'AI API key is not configured.');
        }
        
        // Prepare the prompt with better structure
        $prompt = "You are a professional content writer. Create a high-quality blog post based on these keywords: " . $keywords . "\n\n";
        
        if (!empty($knowledge_base)) {
            $prompt .= "Use this knowledge base for context and expertise:\n\n" . $knowledge_base . "\n\n";
        }
        
        $prompt .= "REQUIREMENTS:\n";
        $prompt .= "1. Create a compelling, SEO-optimized title (50-60 characters)\n";
        $prompt .= "2. Write engaging, informative content (800-1200 words)\n";
        $prompt .= "3. Use proper heading structure (H2, H3)\n";
        $prompt .= "4. Include relevant keywords naturally\n";
        $prompt .= "5. Add a compelling introduction and conclusion\n";
        $prompt .= "6. Make it valuable and actionable for readers\n";
        $prompt .= "7. Use clear, professional language\n";
        $prompt .= "8. Include 2-3 relevant subheadings\n";
        $prompt .= "9. End with a call-to-action\n\n";
        $prompt .= "FORMAT YOUR RESPONSE EXACTLY AS:\n";
        $prompt .= "TITLE: [Your SEO-optimized title here]\n";
        $prompt .= "CONTENT: [Your well-structured blog content here]";
        
        // Retry mechanism for API calls
        $max_retries = 3;
        $retry_delay = 2;
        $response = null;
        
        for ($attempt = 1; $attempt <= $max_retries; $attempt++) {
            $response = wp_remote_post('https://api.openai.com/v1/chat/completions', array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $api_key,
                    'Content-Type' => 'application/json'
                ),
                'body' => json_encode(array(
                    'model' => $model,
                    'messages' => array(
                        array('role' => 'system', 'content' => 'You are a professional blog writer with expertise in creating engaging, SEO-optimized content.'),
                        array('role' => 'user', 'content' => $prompt)
                    ),
                    'max_tokens' => 2500,
                    'temperature' => 0.7,
                    'top_p' => 0.9,
                    'frequency_penalty' => 0.1,
                    'presence_penalty' => 0.1
                )),
                'timeout' => 90
            ));
            
            if (!is_wp_error($response)) {
                break;
            }
            
            if ($attempt < $max_retries) {
                error_log("CGAP: API call attempt $attempt failed, retrying in $retry_delay seconds...");
                sleep($retry_delay);
                $retry_delay *= 2; // Exponential backoff
            }
        }
        
        if (is_wp_error($response)) {
            return array('success' => false, 'error' => 'Error connecting to AI service: ' . $response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!isset($data['choices'][0]['message']['content'])) {
            return array('success' => false, 'error' => 'Error generating content from AI service.');
        }
        
        $content = $data['choices'][0]['message']['content'];
        
        // Parse title and content with better error handling
        preg_match('/TITLE:\s*(.*?)\s*CONTENT:\s*(.*)/s', $content, $matches);
        
        if (count($matches) < 3) {
            // Fallback: use keywords as title
            $title = ucwords(str_replace(',', ' ', $keywords));
            $content = $matches[1] ?? $content;
            
            // Log the parsing issue for debugging
            error_log('CGAP: Content parsing failed, using fallback. Raw content: ' . substr($content, 0, 200) . '...');
        } else {
            $title = trim($matches[1]);
            $content = trim($matches[2]);
        }
        
        // Validate content quality
        if (strlen($content) < 500) {
            return array('success' => false, 'error' => 'Generated content is too short. Please try again with different keywords.');
        }
        
        if (strlen($title) < 10 || strlen($title) > 100) {
            $title = ucwords(str_replace(',', ' ', $keywords)); // Use keywords as fallback title
        }
        
        // Create post
        $post_data = array(
            'post_title' => $title,
            'post_content' => $content,
            'post_status' => $settings['post_status'] ?? 'draft',
            'post_author' => get_current_user_id(),
            'post_type' => 'post'
        );
        
        if (!empty($settings['post_category'])) {
            $post_data['post_category'] = array($settings['post_category']);
        }
        
        $post_id = wp_insert_post($post_data);
        
        if (is_wp_error($post_id)) {
            return array('success' => false, 'error' => 'Error creating post: ' . $post_id->get_error_message());
        }
        
        // Add tags
        if (!empty($settings['post_tags'])) {
            wp_set_post_tags($post_id, $settings['post_tags']);
        }
        
        // Generate and set featured image if premium
        $image_url = '';
        if ($this->is_license_active() && $settings['generate_images']) {
            $image_result = $this->generate_ai_image($title, $settings['image_style'] ?? 'realistic');
            if ($image_result['success']) {
                $image_url = $image_result['url'];
                $this->set_featured_image($post_id, $image_url);
            }
        }
        
        // Save to database
        global $wpdb;
        $table_name = $wpdb->prefix . 'cgap_auto_posts';
        $wpdb->insert($table_name, array(
            'post_id' => $post_id,
            'keywords' => $keywords,
            'generated_content' => $content,
            'image_url' => $image_url
        ));
        
        return array(
            'success' => true,
            'post_id' => $post_id,
            'title' => $title,
            'url' => get_permalink($post_id),
            'image_url' => $image_url
        );
    }
    
    /**
     * Generate AI image
     */
    private function generate_ai_image($prompt, $style = 'realistic') {
        $settings = get_option('cgap_settings', array());
        $api_key = $settings['ai_api_key'] ?? '';
        
        if (empty($api_key)) {
            return array('success' => false, 'error' => 'AI API key is not configured.');
        }
        
        // Enhance prompt with better style descriptions
        $style_descriptions = array(
            'realistic' => 'photorealistic, professional photography, high resolution, detailed',
            'artistic' => 'artistic, creative, vibrant colors, modern art style',
            'minimalist' => 'minimalist, clean, simple, modern design',
            'vintage' => 'vintage, retro, classic, nostalgic',
            'modern' => 'modern, contemporary, sleek, professional design'
        );
        
        $style_desc = $style_descriptions[$style] ?? $style_descriptions['realistic'];
        $enhanced_prompt = "Professional, high-quality image of: " . $prompt . ". Style: " . $style_desc . ". No text, no watermarks, clean composition.";
        
        // Retry mechanism for image generation
        $max_retries = 2;
        $retry_delay = 3;
        $response = null;
        
        for ($attempt = 1; $attempt <= $max_retries; $attempt++) {
            $response = wp_remote_post('https://api.openai.com/v1/images/generations', array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $api_key,
                    'Content-Type' => 'application/json'
                ),
                'body' => json_encode(array(
                    'prompt' => $enhanced_prompt,
                    'n' => 1,
                    'size' => '1024x1024',
                    'response_format' => 'url',
                    'quality' => 'hd'
                )),
                'timeout' => 90
            ));
            
            if (!is_wp_error($response)) {
                break;
            }
            
            if ($attempt < $max_retries) {
                error_log("CGAP: Image generation attempt $attempt failed, retrying in $retry_delay seconds...");
                sleep($retry_delay);
            }
        }
        
        if (is_wp_error($response)) {
            return array('success' => false, 'error' => 'Error connecting to AI image service: ' . $response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!isset($data['data'][0]['url'])) {
            return array('success' => false, 'error' => 'Error generating image from AI service.');
        }
        
        return array(
            'success' => true,
            'url' => $data['data'][0]['url']
        );
    }
    
    /**
     * Set featured image
     */
    private function set_featured_image($post_id, $image_url) {
        // Download image
        $upload_dir = wp_upload_dir();
        $image_data = wp_remote_get($image_url);
        
        if (is_wp_error($image_data)) {
            return false;
        }
        
        $image_content = wp_remote_retrieve_body($image_data);
        $filename = 'auto-poster-' . time() . '.jpg';
        $file_path = $upload_dir['path'] . '/' . $filename;
        
        file_put_contents($file_path, $image_content);
        
        // Prepare file array
        $file = array(
            'name' => $filename,
            'type' => 'image/jpeg',
            'tmp_name' => $file_path,
            'error' => 0,
            'size' => filesize($file_path)
        );
        
        // Upload to media library
        $attachment_id = media_handle_sideload($file, $post_id);
        
        if (is_wp_error($attachment_id)) {
            return false;
        }
        
        // Set as featured image
        set_post_thumbnail($post_id, $attachment_id);
        
        return true;
    }
    
    /**
     * Add powered by footer
     */
    public function powered_by_footer() {
        $screen = get_current_screen();
        if ($screen && strpos($screen->id, 'auto-poster-ai') !== false) {
            echo '<div class="cgap-powered-by" style="position: fixed; bottom: 20px; right: 20px; z-index: 9999; font-family: -apple-system, BlinkMacSystemFont, \'Segoe UI\', Roboto, sans-serif; color: #6b7280; font-size: 0.9rem;">';
            echo '<span>Powered by</span> ';
            echo '<a href="https://codegrinders.tech" target="_blank" style="color: #D84040; font-size: 1.5rem; font-weight: 700; text-decoration: none;">CodeGrinders.tech</a>';
            echo '</div>';
        }
    }

    /**
     * Get license expiration info
     */
    public function get_license_expiration_info() {
        $settings = get_option('cgap_settings', array());
        $expires_at = $settings['license_expires_at'] ?? null;
        
        if (!$expires_at) {
            return array('has_expiration' => false);
        }
        
        $expiration_time = strtotime($expires_at);
        $current_time = time();
        $days_until_expiry = ceil(($expiration_time - $current_time) / (24 * 60 * 60));
        
        return array(
            'has_expiration' => true,
            'expires_at' => $expires_at,
            'days_until_expiry' => $days_until_expiry,
            'is_expired' => $days_until_expiry <= 0,
            'is_expiring_soon' => $days_until_expiry <= 30 && $days_until_expiry > 0
        );
    }
}

// Initialize the plugin
$cgap_plugin = new CodeGrindersAutoPoster();

// Add powered by footer to admin pages
if (!has_action('admin_footer', array($cgap_plugin, 'powered_by_footer'))) {
    add_action('admin_footer', array($cgap_plugin, 'powered_by_footer'));
}
?> 