/**
 * CodeGrinders Auto Poster AI Admin JavaScript
 */

(function($) {
    'use strict';

    // Global variables
    let currentModal = null;
    let isGenerating = false;

    // Initialize when document is ready
    $(document).ready(function() {
        initModals();
        initDashboard();
        initSettings();
        initKnowledge();
        initPosts();
        initCharacterCount();
    });

    /**
     * Initialize modal functionality
     */
    function initModals() {
        // Open modal
        $(document).on('click', '[data-modal]', function(e) {
            e.preventDefault();
            const modalId = $(this).data('modal');
            openModal(modalId);
        });

        // Close modal
        $(document).on('click', '.cgap-modal-close, .cgap-modal', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });

        // Close modal with escape key
        $(document).on('keydown', function(e) {
            if (e.key === 'Escape' && currentModal) {
                closeModal();
            }
        });
    }

    /**
     * Open modal
     */
    function openModal(modalId) {
        currentModal = $('#' + modalId);
        currentModal.addClass('show');
        $('body').addClass('modal-open');
        
        // Focus first input
        setTimeout(() => {
            currentModal.find('input, textarea').first().focus();
        }, 100);
    }

    /**
     * Close modal
     */
    function closeModal() {
        if (currentModal) {
            currentModal.removeClass('show');
            currentModal = null;
            $('body').removeClass('modal-open');
        }
    }

    /**
     * Initialize dashboard functionality
     */
    function initDashboard() {
        // Generate post button
        $(document).on('click', '#generate-post-btn, #generate-first-post-btn', function() {
            openModal('generate-post-modal');
        });

        // Generate image button
        $(document).on('click', '#generate-image-btn', function() {
            openModal('generate-image-modal');
        });

        // Upload knowledge button
        $(document).on('click', '#upload-knowledge-btn', function() {
            openModal('upload-knowledge-modal');
        });

        // Upgrade button
        $(document).on('click', '#upgrade-btn', function() {
            window.open('https://codegrinders.tech', '_blank');
        });

        // Confirm generate post
        $(document).on('click', '#confirm-generate', function() {
            generatePost();
        });

        // Cancel generate post
        $(document).on('click', '#cancel-generate', function() {
            closeModal();
        });

        // Confirm generate image
        $(document).on('click', '#confirm-image', function() {
            generateImage();
        });

        // Cancel generate image
        $(document).on('click', '#cancel-image', function() {
            closeModal();
        });

        // Confirm upload knowledge
        $(document).on('click', '#confirm-knowledge', function() {
            uploadKnowledge();
        });

        // Cancel upload knowledge
        $(document).on('click', '#cancel-knowledge', function() {
            closeModal();
        });
    }

    /**
     * Initialize settings functionality
     */
    function initSettings() {
        // Save settings
        $(document).on('click', '#save-settings-btn', function() {
            saveSettings();
        });

        // Validate license
        $(document).on('click', '#validate-license-btn', function() {
            validateLicense();
        });
        
        // Check license
        $(document).on('click', '#check-license-btn', function() {
            checkLicense();
        });

        // Test connection
        $(document).on('click', '#test-connection-btn', function() {
            openModal('test-connection-modal');
        });

        // Reset settings
        $(document).on('click', '#reset-settings-btn', function() {
            if (confirm('Are you sure you want to reset all settings to defaults?')) {
                resetSettings();
            }
        });

        // Confirm test connection
        $(document).on('click', '#confirm-test', function() {
            testConnection();
        });

        // Cancel test connection
        $(document).on('click', '#cancel-test', function() {
            closeModal();
        });
    }

    /**
     * Initialize knowledge base functionality
     */
    function initKnowledge() {
        // Save knowledge base
        $(document).on('click', '#save-knowledge-btn', function() {
            saveKnowledgeBase();
        });

        // Clear knowledge base
        $(document).on('click', '#clear-knowledge-btn', function() {
            if (confirm('Are you sure you want to clear the knowledge base? This action cannot be undone.')) {
                clearKnowledgeBase();
            }
        });

        // Export knowledge base
        $(document).on('click', '#export-knowledge-btn', function() {
            exportKnowledgeBase();
        });

        // Import knowledge base
        $(document).on('click', '#import-knowledge-btn', function() {
            openModal('import-knowledge-modal');
        });

        // Test knowledge base
        $(document).on('click', '#test-knowledge-btn', function() {
            openModal('test-knowledge-modal');
        });

        // Upgrade knowledge
        $(document).on('click', '#upgrade-knowledge-btn', function() {
            window.open('https://codegrinders.tech', '_blank');
        });

        // Confirm import
        $(document).on('click', '#confirm-import', function() {
            importKnowledgeBase();
        });

        // Cancel import
        $(document).on('click', '#cancel-import', function() {
            closeModal();
        });

        // Confirm test knowledge
        $(document).on('click', '#confirm-test-knowledge', function() {
            testKnowledgeBase();
        });

        // Cancel test knowledge
        $(document).on('click', '#cancel-test-knowledge', function() {
            closeModal();
        });
    }

    /**
     * Initialize posts functionality
     */
    function initPosts() {
        // Generate new post
        $(document).on('click', '#generate-new-post-btn', function() {
            openModal('generate-post-modal');
        });

        // Export posts
        $(document).on('click', '#export-posts-btn', function() {
            exportPosts();
        });

        // Regenerate post
        $(document).on('click', '.regenerate-btn', function() {
            const postId = $(this).data('post-id');
            regeneratePost(postId);
        });

        // Delete post
        $(document).on('click', '.delete-btn', function() {
            const postId = $(this).data('post-id');
            const postTitle = $(this).closest('tr').find('.post-title-cell strong').text();
            deletePost(postId, postTitle);
        });

        // Confirm delete
        $(document).on('click', '#confirm-delete', function() {
            const postId = $('#delete-post-id').text();
            confirmDeletePost(postId);
        });

        // Cancel delete
        $(document).on('click', '#cancel-delete', function() {
            closeModal();
        });

        // Confirm regenerate
        $(document).on('click', '#confirm-regenerate', function() {
            const postId = $('#regenerate-post-id').text();
            confirmRegeneratePost(postId);
        });

        // Cancel regenerate
        $(document).on('click', '#cancel-regenerate', function() {
            closeModal();
        });
    }

    /**
     * Initialize character count
     */
    function initCharacterCount() {
        $(document).on('input', '#knowledge-text', function() {
            updateCharacterCount();
        });
    }

    /**
     * Update character count
     */
    function updateCharacterCount() {
        const text = $('#knowledge-text').val();
        const count = text.length;
        $('#char-count').text(count.toLocaleString());
        
        // Update progress bar
        const maxChars = 5000; // This should come from PHP
        const percentage = Math.min((count / maxChars) * 100, 100);
        $('.char-count-fill').css('width', percentage + '%');
    }

    /**
     * Generate post
     */
    function generatePost() {
        if (isGenerating) return;
        
        const keywords = $('#post-keywords').val().trim();
        const generateImage = $('#generate-image-checkbox').is(':checked');
        
        if (!keywords) {
            showMessage('Please enter keywords for post generation.', 'error');
            return;
        }

        isGenerating = true;
        const $btn = $('#confirm-generate');
        const originalText = $btn.html();
        $btn.html('<i class="dashicons dashicons-update"></i> Generating...').prop('disabled', true);

        $.ajax({
            url: cgap_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'cgap_generate_post',
                nonce: cgap_ajax.nonce,
                keywords: keywords,
                generate_image: generateImage
            },
            success: function(response) {
                if (response.success) {
                    showMessage('Post generated successfully!', 'success');
                    closeModal();
                    setTimeout(() => {
                        location.reload();
                    }, 1500);
                } else {
                    showMessage(response.data || 'Error generating post.', 'error');
                }
            },
            error: function() {
                showMessage('Error connecting to server.', 'error');
            },
            complete: function() {
                isGenerating = false;
                $btn.html(originalText).prop('disabled', false);
            }
        });
    }

    /**
     * Generate image
     */
    function generateImage() {
        if (isGenerating) return;
        
        const prompt = $('#image-prompt').val().trim();
        const style = $('#image-style').val();
        
        if (!prompt) {
            showMessage('Please enter a prompt for image generation.', 'error');
            return;
        }

        isGenerating = true;
        const $btn = $('#confirm-image');
        const originalText = $btn.html();
        $btn.html('<i class="dashicons dashicons-update"></i> Generating...').prop('disabled', true);

        $.ajax({
            url: cgap_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'cgap_generate_image',
                nonce: cgap_ajax.nonce,
                prompt: prompt,
                style: style
            },
            success: function(response) {
                if (response.success) {
                    showMessage('Image generated successfully!', 'success');
                    closeModal();
                    // Show image preview
                    showImagePreview(response.data.url);
                } else {
                    showMessage(response.data || 'Error generating image.', 'error');
                }
            },
            error: function() {
                showMessage('Error connecting to server.', 'error');
            },
            complete: function() {
                isGenerating = false;
                $btn.html(originalText).prop('disabled', false);
            }
        });
    }

    /**
     * Upload knowledge base
     */
    function uploadKnowledge() {
        const knowledgeText = $('#knowledge-text').val().trim();
        
        if (!knowledgeText) {
            showMessage('Please enter knowledge base content.', 'error');
            return;
        }

        $.ajax({
            url: cgap_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'cgap_upload_knowledge',
                nonce: cgap_ajax.nonce,
                knowledge_text: knowledgeText
            },
            success: function(response) {
                if (response.success) {
                    showMessage('Knowledge base updated successfully!', 'success');
                    closeModal();
                    setTimeout(() => {
                        location.reload();
                    }, 1500);
                } else {
                    showMessage(response.data || 'Error updating knowledge base.', 'error');
                }
            },
            error: function() {
                showMessage('Error connecting to server.', 'error');
            }
        });
    }

    /**
     * Save settings
     */
    function saveSettings() {
        const settings = {
            keywords: $('#keywords').val(),
            post_frequency: $('#post-frequency').val(),
            post_status: $('#post-status').val(),
            post_category: $('#post-category').val(),
            post_tags: $('#post-tags').val(),
            generate_images: $('#generate-images').is(':checked'),
            image_style: $('#image-style').val(),
            ai_api_key: $('#ai-api-key').val(),
            ai_model: $('#ai-model').val(),
            is_enabled: $('#is-enabled').is(':checked')
        };

        $.ajax({
            url: cgap_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'cgap_save_settings',
                nonce: cgap_ajax.nonce,
                ...settings
            },
            success: function(response) {
                if (response.success) {
                    showMessage('Settings saved successfully!', 'success');
                } else {
                    showMessage(response.data || 'Error saving settings.', 'error');
                }
            },
            error: function() {
                showMessage('Error connecting to server.', 'error');
            }
        });
    }

    /**
     * Validate license
     */
    function validateLicense() {
        const licenseKey = $('#license-key').val().trim();
        
        if (!licenseKey) {
            showMessage('Please enter a license key.', 'error');
            return;
        }

        const $btn = $('#validate-license-btn');
        const originalText = $btn.html();
        $btn.html('<i class="dashicons dashicons-update"></i> Validating...').prop('disabled', true);

        $.ajax({
            url: cgap_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'cgap_validate_license',
                nonce: cgap_ajax.nonce,
                license_key: licenseKey
            },
            success: function(response) {
                if (response.success) {
                    showMessage('License validated successfully!', 'success');
                    setTimeout(() => {
                        location.reload();
                    }, 1500);
                } else {
                    showMessage(response.data || 'License validation failed.', 'error');
                }
            },
            error: function() {
                showMessage('Error connecting to server.', 'error');
            },
            complete: function() {
                $btn.html(originalText).prop('disabled', false);
            }
        });
    }
    
    /**
     * Check license status
     */
    function checkLicense() {
        const $btn = $('#check-license-btn');
        const originalText = $btn.html();
        $btn.html('<i class="dashicons dashicons-update"></i> Checking...').prop('disabled', true);

        $.ajax({
            url: cgap_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'cgap_check_license',
                nonce: cgap_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    showMessage('License is valid and active!', 'success');
                    setTimeout(() => {
                        location.reload();
                    }, 1500);
                } else {
                    showMessage(response.data || 'License check failed.', 'error');
                }
            },
            error: function() {
                showMessage('Error connecting to server.', 'error');
            },
            complete: function() {
                $btn.html(originalText).prop('disabled', false);
            }
        });
    }

    /**
     * Test connection
     */
    function testConnection() {
        const apiKey = $('#test-api-key').val().trim();
        const model = $('#test-model').val();
        
        if (!apiKey) {
            showMessage('Please enter an API key for testing.', 'error');
            return;
        }

        const $btn = $('#confirm-test');
        const originalText = $btn.html();
        $btn.html('<i class="dashicons dashicons-update"></i> Testing...').prop('disabled', true);

        // Simulate test (in real implementation, this would call the AI API)
        setTimeout(() => {
            showMessage('Connection test successful!', 'success');
            closeModal();
            $btn.html(originalText).prop('disabled', false);
        }, 2000);
    }

    /**
     * Reset settings
     */
    function resetSettings() {
        $.ajax({
            url: cgap_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'cgap_reset_settings',
                nonce: cgap_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    showMessage('Settings reset successfully!', 'success');
                    setTimeout(() => {
                        location.reload();
                    }, 1500);
                } else {
                    showMessage(response.data || 'Error resetting settings.', 'error');
                }
            },
            error: function() {
                showMessage('Error connecting to server.', 'error');
            }
        });
    }

    /**
     * Save knowledge base
     */
    function saveKnowledgeBase() {
        const knowledgeText = $('#knowledge-text').val().trim();
        
        if (!knowledgeText) {
            showMessage('Please enter knowledge base content.', 'error');
            return;
        }

        $.ajax({
            url: cgap_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'cgap_upload_knowledge',
                nonce: cgap_ajax.nonce,
                knowledge_text: knowledgeText
            },
            success: function(response) {
                if (response.success) {
                    showMessage('Knowledge base saved successfully!', 'success');
                    updateCharacterCount();
                } else {
                    showMessage(response.data || 'Error saving knowledge base.', 'error');
                }
            },
            error: function() {
                showMessage('Error connecting to server.', 'error');
            }
        });
    }

    /**
     * Clear knowledge base
     */
    function clearKnowledgeBase() {
        $('#knowledge-text').val('');
        updateCharacterCount();
        showMessage('Knowledge base cleared successfully!', 'success');
    }

    /**
     * Export knowledge base
     */
    function exportKnowledgeBase() {
        const knowledgeText = $('#knowledge-text').val();
        const blob = new Blob([knowledgeText], { type: 'text/plain' });
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = 'knowledge-base.txt';
        a.click();
        window.URL.revokeObjectURL(url);
    }

    /**
     * Import knowledge base
     */
    function importKnowledgeBase() {
        const fileInput = $('#import-file')[0];
        const replaceExisting = $('#replace-existing').is(':checked');
        
        if (!fileInput.files.length) {
            showMessage('Please select a file to import.', 'error');
            return;
        }

        const file = fileInput.files[0];
        const reader = new FileReader();
        
        reader.onload = function(e) {
            const content = e.target.result;
            if (replaceExisting) {
                $('#knowledge-text').val(content);
            } else {
                const currentContent = $('#knowledge-text').val();
                $('#knowledge-text').val(currentContent + '\n\n' + content);
            }
            updateCharacterCount();
            closeModal();
            showMessage('Knowledge base imported successfully!', 'success');
        };
        
        reader.readAsText(file);
    }

    /**
     * Test knowledge base
     */
    function testKnowledgeBase() {
        const keywords = $('#test-keywords').val().trim();
        const prompt = $('#test-prompt').val().trim();
        
        if (!keywords) {
            showMessage('Please enter test keywords.', 'error');
            return;
        }

        const $btn = $('#confirm-test-knowledge');
        const originalText = $btn.html();
        $btn.html('<i class="dashicons dashicons-update"></i> Testing...').prop('disabled', true);

        // Simulate test (in real implementation, this would call the AI API)
        setTimeout(() => {
            showMessage('Knowledge base test completed successfully!', 'success');
            closeModal();
            $btn.html(originalText).prop('disabled', false);
        }, 2000);
    }

    /**
     * Export posts
     */
    function exportPosts() {
        // This would typically make an AJAX call to get posts data
        showMessage('Export functionality coming soon!', 'warning');
    }

    /**
     * Regenerate post
     */
    function regeneratePost(postId) {
        // Get post data and open regenerate modal
        $('#regenerate-post-id').text(postId);
        $('#regenerate-keywords').text('Keywords from original post'); // This should be fetched from server
        openModal('regenerate-post-modal');
    }

    /**
     * Delete post
     */
    function deletePost(postId, postTitle) {
        $('#delete-post-id').text(postId);
        $('#delete-post-title').text(postTitle);
        openModal('delete-post-modal');
    }

    /**
     * Confirm delete post
     */
    function confirmDeletePost(postId) {
        $.ajax({
            url: cgap_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'cgap_delete_post',
                nonce: cgap_ajax.nonce,
                post_id: postId
            },
            success: function(response) {
                if (response.success) {
                    showMessage('Post deleted successfully!', 'success');
                    closeModal();
                    setTimeout(() => {
                        location.reload();
                    }, 1500);
                } else {
                    showMessage(response.data || 'Error deleting post.', 'error');
                }
            },
            error: function() {
                showMessage('Error connecting to server.', 'error');
            }
        });
    }

    /**
     * Confirm regenerate post
     */
    function confirmRegeneratePost(postId) {
        const generateImage = $('#regenerate-image-checkbox').is(':checked');
        
        $.ajax({
            url: cgap_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'cgap_regenerate_post',
                nonce: cgap_ajax.nonce,
                post_id: postId,
                generate_image: generateImage
            },
            success: function(response) {
                if (response.success) {
                    showMessage('Post regenerated successfully!', 'success');
                    closeModal();
                    setTimeout(() => {
                        location.reload();
                    }, 1500);
                } else {
                    showMessage(response.data || 'Error regenerating post.', 'error');
                }
            },
            error: function() {
                showMessage('Error connecting to server.', 'error');
            }
        });
    }

    /**
     * Show image preview
     */
    function showImagePreview(imageUrl) {
        const modal = `
            <div class="cgap-modal show" id="image-preview-modal">
                <div class="cgap-modal-content">
                    <div class="cgap-modal-header">
                        <h2><i class="dashicons dashicons-format-image"></i> Generated Image</h2>
                        <button type="button" class="cgap-modal-close">&times;</button>
                    </div>
                    <div class="cgap-modal-body">
                        <img src="${imageUrl}" alt="Generated image" style="width: 100%; border-radius: 8px;">
                    </div>
                    <div class="cgap-modal-footer">
                        <button type="button" class="cgap-btn cgap-btn-secondary" onclick="closeModal()">Close</button>
                        <a href="${imageUrl}" download class="cgap-btn cgap-btn-primary">
                            <i class="dashicons dashicons-download"></i>
                            Download Image
                        </a>
                    </div>
                </div>
            </div>
        `;
        $('body').append(modal);
    }

    /**
     * Show message
     */
    function showMessage(message, type = 'info') {
        const messageHtml = `
            <div class="cgap-message ${type}">
                ${message}
                <button type="button" class="cgap-message-close" style="float: right; background: none; border: none; font-size: 1.2rem; cursor: pointer;">&times;</button>
            </div>
        `;
        
        // Remove existing messages
        $('.cgap-message').remove();
        
        // Add new message
        $('.wrap').first().prepend(messageHtml);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            $('.cgap-message').fadeOut();
        }, 5000);
        
        // Close button
        $(document).on('click', '.cgap-message-close', function() {
            $(this).parent().fadeOut();
        });
    }

    // Make functions globally available
    window.openModal = openModal;
    window.closeModal = closeModal;
    window.showMessage = showMessage;

})(jQuery);
